"""
DEMVis - Digital Elevation Model's Visualization (Build 2009-12-07)

The DEMVis (Digital Elevation Model's Visualization is a open source software to
visualization of USGS DEM files.

Created in December 2009 by Rodrigo Mologni G dos Santos
<mologni@dca.fee.unicamp.br> and Wagner Machado do Amaral
<machado@dca.fee.unicamp.br>.
"""

import vtk

from vtk.qt4.QVTKRenderWindowInteractor import QVTKRenderWindowInteractor

class QVTKWidget():

    SCALE = 5

    def __init__(self, parent):
        self.parent       = parent
        self.dem          = vtk.vtkDEMReader()
        self.colormap     = vtk.vtkLookupTable()
        self.shrink       = vtk.vtkImageShrink3D()
        self.geometry     = vtk.vtkImageDataGeometryFilter()
        self.warp         = vtk.vtkWarpScalar()
        self.elevation    = vtk.vtkElevationFilter()
        self.band         = vtk.vtkBandedPolyDataContourFilter()
        self.normals      = vtk.vtkPolyDataNormals()
        self.demMapper    = vtk.vtkPolyDataMapper()
        self.demActor     = vtk.vtkLODActor()
        self.edgeMapper   = vtk.vtkPolyDataMapper()
        self.edgeActor    = vtk.vtkActor()
        self.textureActor = vtk.vtkActor()
        self.barActor     = vtk.vtkScalarBarActor()
        self.barIsVisible = False
    
        # The camera object controls how 3D geometry is projected into the 2D
        # image plane during the rendering process.
        self.camera = vtk.vtkCamera()

        # The renderer draws.
        self.render = vtk.vtkRenderer()
        self.render.AddActor(self.demActor)
        self.render.SetActiveCamera(self.camera)
        self.render.SetBackground(0.4, 0.4, 0.4)

        # A simple tool for manipulating the camera, picking objects, invoking
        # user-defined methods, entering/exiting stereo viewing, and changing
        # some of the properties of actors.
        self.widget = QVTKRenderWindowInteractor(self.parent)
        self.widget.GetRenderWindow().AddRenderer(self.render)

        self.parent.setCentralWidget(self.widget)
    
    # Set file name...
    
    def setFileName(self, fileName):
        # Read Digital Elevation Model (DEM) files.
        self.dem.SetFileName(str(fileName))
        self.dem.Update()
        self.draw(self.parent)
    
    # Draw
    
    def draw(self, parent):
        # Defines colors and the range of data values.
        self.setLookupTableToLand()

        # Shrink (reduce the extent of) and image by subsampling on a uniform
        # grid.
        self.setShrinkFactorsToMaximum()
        self.shrink.SetInputConnection(self.dem.GetOutputPort())
        self.shrink.AveragingOn()
    
        # Extract geometry (points, lines, planes) as vtkPolyData (only
        # polygonal data).
        self.geometry.SetInputConnection(self.shrink.GetOutputPort())
        
        # Modify point coordinates by scaling according to scalar values.
        self.warp.SetInputConnection(self.geometry.GetOutputPort())
        self.warp.UseNormalOn()
        self.warp.SetScaleFactor(self.SCALE)
        self.setWarpScalarNormalTo2D()
        
        # Generate scalars according to projection along a vector.
        self.low  = self.SCALE * self.dem.GetElevationBounds()[0]
        self.high = self.SCALE * self.dem.GetElevationBounds()[1]
        self.elevation.SetInputConnection(self.warp.GetOutputPort())
        self.elevation.SetLowPoint(0, 0, self.low)
        self.elevation.SetHighPoint(0, 0, self.high)
        self.elevation.SetScalarRange(self.low, self.high)
        self.elevation.ReleaseDataFlagOn()

        # Generate filled contours (bands of cells that all have the same cell
        # scalar value).
        self.band.SetInput(self.warp.GetPolyDataOutput())
        self.band.GenerateValues(15, self.dem.GetOutput().GetScalarRange())
        self.band.SetScalarModeToIndex()
        self.band.GenerateContourEdgesOn()
        
        # Generate surface normal vectors (i.e., vectors perpendicular to the
        # geometric surface) at each point in the dataset.
        self.normals.SetInputConnection(self.elevation.GetOutputPort())
        self.normals.SetFeatureAngle(60)
        self.normals.ConsistencyOff()
        self.normals.SplittingOff()
        
        # Maps polygonal data to the graphics system.
        self.demMapper.SetInputConnection(self.normals.GetOutputPort())
        self.demMapper.SetScalarRange(self.low, self.high)
        self.demMapper.SetLookupTable(self.colormap)
        
        # A simple level-of-detail scheme for rendering 3D geometry.
        self.demActor.SetMapper(self.demMapper)
        
        # A labeled, colored bar that visually expresses the relationship
        # between color and scalar value.
        self.barActor.SetLookupTable(self.demMapper.GetLookupTable())
        self.barActor.SetTextPositionToPrecedeScalarBar()
    
        # Draws vtkPolyData into the overlay plane.
        self.edgeMapper.SetInput(self.band.GetContourEdgesOutput())
        self.edgeMapper.SetResolveCoincidentTopologyToPolygonOffset()
        
        # A type of vtkProp3D whose geometry is defined by analytic primitives
        # such as polygons and lines; it is often used for representing a
        # dataset in a 3D scene.
        self.edgeActor.SetMapper(self.edgeMapper)
        self.edgeActor.GetProperty().SetColor(0, 0, 0)

        # Initialize all components
        self.setCameraPositionToZPlus()
        self.setStereoTypeToCrystalEyes()
        self.setContourToInvisible()
        self.setColorBandToInvisible()
        self.setColorLegendToInvisible()
        self.setClipToInsivible()

        self.render.ResetCamera()

    # Set lookuptable to...

    def setLookupTable(self, numberOfColors, hueRange, saturationRange,
            valueRange):
        self.colormap.SetNumberOfColors(numberOfColors)
        self.colormap.SetHueRange(hueRange[0], hueRange[1])
        self.colormap.SetSaturationRange(saturationRange[0], saturationRange[1])
        self.colormap.SetValueRange(valueRange[0], valueRange[1])
        self.colormap.Build()
        
        self.render.RemoveActor(self.textureActor)
        self.render.AddActor(self.demActor)
        
        if self.barIsVisible == True:
            self.render.AddActor(self.barActor)
    
    def setLookupTableToLuminance(self):
        self.setLookupTable(256, [0.0, 0.0], [0.0, 0.0], [0.334, 1.0])
    
    def setLookupTableToRainbow(self):
        self.setLookupTable(256, [0.667, 0.0], [1.0, 1.0], [1.0, 1.0])
    
    def setLookupTableToWater(self):
        self.setLookupTable(256, [0.58, 0.58], [0.5, 0.1], [0.5, 1.0])
    
    def setLookupTableToLand(self):
        self.setLookupTable(256, [0.1, 0.1], [0.4, 0.1], [0.55, 0.9])
    
    def setLookupTableToReality(self):
        self.setLookupTable(256, [0.6, 0.0], [1.0, 0.0], [0.5, 1.0])

    # Set shrink factors to...
    
    def setShrinkFactors(self, value):
        self.shrink.SetShrinkFactors(value, value, 1)
    
    def setShrinkFactorsToMinimum(self):
        self.setShrinkFactors(0)

    def setShrinkFactorsToMedium(self):
        self.setShrinkFactors(2)
    
    def setShrinkFactorsToMaximum(self):
        self.setShrinkFactors(4)
    
    # Set camera position to...

    def setCameraPositionToXPlus(self):
        self.camera.SetPosition(1, 0, 0)
        self.camera.SetFocalPoint(0, 0, 0)
        self.camera.SetViewUp(0, 0, 1)
        self.render.ResetCamera()
        self.widget.GetRenderWindow().Render()
    
    def setCameraPositionToXMinus(self):
        self.camera.SetPosition(-1, 0, 0)
        self.camera.SetFocalPoint(0, 0, 0)
        self.camera.SetViewUp(0, 0, 1)
        self.render.ResetCamera()
        self.widget.GetRenderWindow().Render()
    
    def setCameraPositionToYPlus(self):
        self.camera.SetPosition(0, 1, 0)
        self.camera.SetFocalPoint(0, 0, 0)
        self.camera.SetViewUp(0, 0, 1)
        self.render.ResetCamera()
        self.widget.GetRenderWindow().Render()
    
    def setCameraPositionToYMinus(self):
        self.camera.SetPosition(0, -1, 0)
        self.camera.SetFocalPoint(0, 0, 0)
        self.camera.SetViewUp(0, 0, 1)
        self.render.ResetCamera()
        self.widget.GetRenderWindow().Render()
    
    def setCameraPositionToZPlus(self):
        self.camera.SetPosition(0, 0, 1)
        self.camera.SetFocalPoint(0, 0, 0)
        self.camera.SetViewUp(1, 0, 0)
        self.render.ResetCamera()
        self.widget.GetRenderWindow().Render()
    
    def setCameraPositionToZMinus(self):
        self.camera.SetPosition(0, 0, -1)
        self.camera.SetFocalPoint(0, 0, 0)
        self.camera.SetViewUp(1, 0, 0)
        self.render.ResetCamera()
        self.widget.GetRenderWindow().Render()
        
    # Set warp scalar normal to...
    
    def setWarpScalarNormalTo2D(self):
        self.warp.SetNormal(0, 0, 0)
        self.render.ResetCamera()
        self.widget.GetRenderWindow().Render()
    
    def setWarpScalarNormalTo3D(self):
        self.warp.SetNormal(0, 0, 1)
        self.render.ResetCamera()
        self.widget.GetRenderWindow().Render()
        
    # Set stereo type to...
    
    def setStereoTypeToAnaglyph(self):
        self.widget.GetRenderWindow().SetStereoTypeToAnaglyph()
        self.widget.GetRenderWindow().StereoRenderOn()
        self.widget.GetRenderWindow().StereoUpdate()
        self.widget.GetRenderWindow().Render()

    def setStereoTypeToCrystalEyes(self):
        self.widget.GetRenderWindow().StereoRenderOff()
        self.widget.GetRenderWindow().SetStereoTypeToCrystalEyes()
        self.widget.GetRenderWindow().StereoUpdate()
        self.widget.GetRenderWindow().Render()
    
    # Set contour to...
    
    def setContourToVisible(self):
        self.render.AddActor(self.edgeActor)
        self.widget.GetRenderWindow().Render()
    
    def setContourToInvisible(self):
        self.render.RemoveActor(self.edgeActor)
        self.widget.GetRenderWindow().Render()
    
    # Set color band to...
    
    def setColorBandToVisible(self):
        self.barActor.SetMaximumNumberOfColors(10)
        self.normals.SetInputConnection(self.band.GetOutputPort())
        self.demMapper.SetScalarRange(0, 10)
        self.demMapper.SetScalarModeToUseCellData()
        self.widget.GetRenderWindow().Render()
    
    def setColorBandToInvisible(self):
        self.barActor.SetMaximumNumberOfColors(256)
        self.normals.SetInputConnection(self.elevation.GetOutputPort())
        self.demMapper.SetScalarRange(self.low, self.high)
        self.demMapper.SetScalarModeToDefault()
        self.widget.GetRenderWindow().Render()
        
    # Capture window

    def captureWindow(self, fileName):
        image = vtk.vtkWindowToImageFilter()
        image.SetInput(self.render.GetRenderWindow())
        image.ReadFrontBufferOff()
        image.Update()

        writer = vtk.vtkPNGWriter()
        writer.SetInput(image.GetOutput())
        writer.SetFileName(str(fileName))
        writer.Write()
    
    # Set texture to...
    
    def setTexture(self, fileName):
        reader = vtk.vtkJPEGReader()
        reader.SetFileName(fileName)
        
        texture = vtk.vtkTexture()
        texture.SetInput(reader.GetOutput())
        
        plane = vtk.vtkTextureMapToPlane()
        plane.SetInput(self.normals.GetOutput())
        
        mapper = vtk.vtkPolyDataMapper()
        mapper.SetInputConnection(plane.GetOutputPort())
        mapper.ScalarVisibilityOff()
        
        self.textureActor.SetMapper(mapper)
        self.textureActor.SetTexture(texture)

        self.render.RemoveActor(self.demActor)
        self.render.RemoveActor(self.barActor)
        self.render.AddActor(self.textureActor)
    
    def setTextureToGlacier(self):
        self.setTexture("../images/textures/glacier.jpg")
    
    def setTextureToDesert(self):
        self.setTexture("../images/textures/desert.jpg")
    
    def setTextureToJungle(self):
        self.setTexture("../images/textures/jungle.jpg")
    
    def setTextureToConcrete(self):
        self.setTexture("../images/textures/concrete.jpg")
    
    # Set color legend to...
    
    def setColorLegendToVisible(self):
        self.barIsVisible = True
        self.render.AddActor(self.barActor)
        self.render.GetRenderWindow().Render()

    def setColorLegendToInvisible(self):
        self.barIsVisible = False
        self.render.RemoveActor(self.barActor)
        self.render.GetRenderWindow().Render()
    
    # Set clip to...

    def setClipToVisible(self):
        self.band.ClippingOn()
        self.band.GenerateValues(10, 1000, 2000)
        self.render.GetRenderWindow().Render()
    
    def setClipToInsivible(self):
        self.band.ClippingOff()
        self.band.GenerateValues(15, self.dem.GetOutput().GetScalarRange())
        self.band.SetScalarModeToIndex()
        self.render.GetRenderWindow().Render()
