//***********************************************************************//
//                                                                       //
//      - "Talk to me like I'm a 3 year old!" Programming Lessons -      //
//                                                                       //
//      $Author:        DigiBen     digiben@gametutorials.com            //
//                                                                       //
//      $Program:       3DS Loader                                       //
//                                                                       //
//      $Description:   Demonstrates how to load a .3ds file format      //
//                                                                       //
//      $Date:          10/6/01                                          //
//                                                                       //
//***********************************************************************//


#include "main.h"                                        // This includes our header file
#include "3ds.h"

#include <math.h>
#include <limits.h>


#define MAX(a,b) (a>b) ? a : b
#define MIN(a,b) (a<b) ? a : b
//////////// *** NEW *** ////////// *** NEW *** ///////////// *** NEW *** ////////////////////

// This tutorial will demonstrate how to load a .3DS file.  This 3D file format
// is created from 3D Studio Max.  It can be exported and important into many
// other programs.  A great tool for converting 3D file formats is "3D Exploration".
// This is a shareware utility that can be found for download on the web.
// This 3DS loader only loads the texture names, object colors, the vertices, the faces, and the UV coordinates.
// The key frame information is ignored since we have a Key Frame Animation tutorial later.
// I didn't want to over complicate this tutorial, but just show the basics.  
//
// In this tutorial, there is a picture of a face rotating around.  It is a single object
// with a single texture.  The .3ds loader does load multiple objects though.
// The controls are: 

// Left Mouse Button - Changes the Render mode from normal to wireframe.
// Right Mouse Button - Turns lighting On/Off
// Left Arrow Key - Spins the model to the left
// Right Arrow Key - Spins the model to the right
// Escape - Quits

///////////////////////////////// INIT GAME WINDOW \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*
/////
/////    This function initializes the game window.
/////
///////////////////////////////// INIT GAME WINDOW \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*


t3DModel::t3DModel () {
    numOfObjects=0;
    numOfMaterials=0;
    m_maxVerts.x = (float)INT_MIN;
    m_maxVerts.y = (float)INT_MIN;
    m_maxVerts.z = (float)INT_MIN;
    m_minVerts.x = (float)INT_MAX;
    m_minVerts.y = (float)INT_MAX;
    m_minVerts.z = (float)INT_MAX;
    for (int i=0;i<MAX_TEXTURES;i++) 
        g_Texture[i] = 0;

}

t3DModel::~t3DModel () {
    for(int i = 0; i < numOfObjects; i++)
    {
        // Free the faces, normals, vertices, and texture coordinates.
        delete [] pObject[i].pFaces;
        delete [] pObject[i].pNormals;
        delete [] pObject[i].pVerts;
        delete [] pObject[i].pTexVerts;
    }
}

void t3DModel::load (char *filename)
{
    CLoad3DS g_Load3ds;                                     // This is 3DS class.  This should go in a good model class.
    g_Load3ds.Import3DS(this, filename);         // Load our .3DS file into our model structure

    // Depending on how many textures we found, load each one (Assuming .BMP)
    // If you want to load other files than bitmaps, you will need to adjust CreateTexture().
    // Below, we go through all of the materials and check if they have a texture map to load.
    // Otherwise, the material just holds the color information and we don't need to load a texture.

    // Go through all the materials
}

void t3DModel::center () {

    for(int i = 0; i < numOfMaterials; i++)
    {
        // Check to see if there is a file name to load in this material
        if(strlen(pMaterials[i].strFile) > 0)
        {
            // Use the name of the texture file to load the bitmap, with a texture ID (i).
            // We pass in our global texture array, the name of the texture, and an ID to reference it. 
            CreateTexture(g_Texture, pMaterials[i].strFile, i);           
        }

        // Set the texture ID for this material
        pMaterials[i].texureId = i;
    }

    // Here, we turn on a lighting and enable lighting.  We don't need to
    // set anything else for lighting because we will just take the defaults.
    // We also want color, so we turn that on

    CVector3  m_transVerts;

    for(int i = 0; i < numOfObjects; i++)
    {
        // Make sure we have valid objects just in case. (size() is in the vector class)
        if(pObject.size() <= 0) break;

        // Get the current object that we are displaying

        for (int j = 0; j < pObject[i].numOfVerts; j++)           // Go through all of the vertices
        {	
          m_minVerts.x = MIN(pObject[i].pVerts[j].x, m_minVerts.x);
          m_maxVerts.x = MAX(pObject[i].pVerts[j].x, m_maxVerts.x);
          m_minVerts.y = MIN(pObject[i].pVerts[j].y, m_minVerts.y);
          m_maxVerts.y = MAX(pObject[i].pVerts[j].y, m_maxVerts.y);
          m_minVerts.z = MIN(pObject[i].pVerts[j].z, m_minVerts.z);
          m_maxVerts.z = MAX(pObject[i].pVerts[j].z, m_maxVerts.z);
	}
    }
    // t.x = -min.x - (max.x-min.x)/2
    m_transVerts.x = -( m_maxVerts.x + m_minVerts.x)/2.;
    m_transVerts.y = -( m_maxVerts.y + m_minVerts.y)/2.;
    m_transVerts.z = -( m_maxVerts.z + m_minVerts.z)/2.;
    for(int i = 0; i < numOfObjects; i++)
      for (int j= 0; j < pObject[i].numOfVerts; j++)           // Go through all of the vertices
        {	
          pObject[i].pVerts[j].x += m_transVerts.x;
          pObject[i].pVerts[j].y += m_transVerts.y;
          pObject[i].pVerts[j].z += m_transVerts.z;
	}   
}

void t3DModel::scale (float sx, float sy, float sz) 
{
    CVector3  m_scale;
    CVector3  m_transVerts;
    CVector3  m_maxOffset;

    m_transVerts.x = -( m_maxVerts.x + m_minVerts.x)/2.;
    m_transVerts.y = -( m_maxVerts.y + m_minVerts.y)/2.;
    m_transVerts.z = -( m_maxVerts.z + m_minVerts.z)/2.;

    m_maxOffset.x = m_maxVerts.x + m_transVerts.x;
    m_maxOffset.y = m_maxVerts.y + m_transVerts.y; 
    m_maxOffset.z = m_maxVerts.z + m_transVerts.z;
    m_scale.x = sx / m_maxOffset.x;
    m_scale.y = sy / m_maxOffset.y;
    m_scale.z = sz / m_maxOffset.z;
  
    for(int i = 0; i < numOfObjects; i++) {
      if(pObject.size() <= 0) break;
      for (int j= 0; j < pObject[i].numOfVerts; j++)           // Go through all of the vertices
        {	
          pObject[i].pVerts[j].x *= m_scale.x;
          pObject[i].pVerts[j].y *= m_scale.y;
          pObject[i].pVerts[j].z *= m_scale.z;
	}
    }
}



/////////////////////////////////////////////////////////////////////////////////
//
// * QUICK NOTES * 
//
// This tutorial shows how to load a .3ds file.  This is a good addition to an engine.
// In the next tutorial we will show you how to load a text file format called .obj.
// This is the most common 3D file format that almost ANY 3D software will import.
// 
// I hope the model loading code wasn't to intimidating.  The first time I read model
// loading code I didn't get the vertex indexing and such, but after careful reading, 
// It all came together.  Please email me if you are confused on ANY of this and I will
// be happy to clarify it until you get it.
//
// * What's An STL (Standard Template Library) Vector? *
// Let me quickly explain the STL vector for those of you who are not familiar with them.
// To use a vector you must include <vector> and use the std namespace: using namespace std;
// A vector is an array based link list.  It allows you to dynamically add and remove nodes.
// This is a template class so it can be a list of ANY type.  To create a vector of type
// "int" you would say:  vector<int> myIntList;
// Now you can add a integer to the dynamic array by saying: myIntList.push_back(10);
// or you can say:  myIntList.push_back(num);.  The more you push back, the larger
// your array gets.  You can index the vector like an array.  myIntList[0] = 0;
// To get rid of a node you use the pop_back() function.  To clear the vector use clear().
// It frees itself so you don't need to worry about it, except if you have data
// structures that need information freed from inside them, like our objects.
//
// Once again I should point out that the coordinate system of OpenGL and 3DS Max are different.
// Since 3D Studio Max Models with the Z-Axis pointing up (strange and ugly I know! :), 
// we need to flip the y values with the z values in our vertices.  That way it
// will be normal, with Y pointing up.  Also, because we swap the Y and Z we need to negate 
// the Z to make it come out correctly.  This is also explained and done in ReadVertices().
//
// Below explains what was explained in 3ds.cpp regarding chucks:
// 
// CHUNKS: What is a chunk anyway?
// 
// "The chunk ID is a unique code which identifies the type of data in this chunk 
// and also may indicate the existence of subordinate chunks. The chunk length indicates 
// the length of following data to be associated with this chunk. Note, this may 
// contain more data than just this chunk. If the length of data is greater than that 
// needed to fill in the information for the chunk, additional subordinate chunks are 
// attached to this chunk immediately following any data needed for this chunk, and 
// should be parsed out. These subordinate chunks may themselves contain subordinate chunks. 
// Unfortunately, there is no indication of the length of data which is owned by the current 
// chunk, only the total length of data attached to the chunk, which means that the only way 
// to parse out subordinate chunks is to know the exact format of the owning chunk. On the 
// other hand, if a chunk is unknown, the parsing program can skip the entire chunk and 
// subordinate chunks in one jump. " - Jeff Lewis (werewolf@worldgate.com)
//
// Below is a list of the order that you will find the chunks and all the know chunks.
// If you go to www.wosit.org you can find the 3DS file format for a huge document explaining
// the little things I skipped.
//
//
//
//      MAIN3DS  (0x4D4D)
//     |
//     +--EDIT3DS  (0x3D3D)
//     |  |
//     |  +--EDIT_MATERIAL (0xAFFF)
//     |  |  |
//     |  |  +--MAT_NAME01 (0xA000) (See mli Doc) 
//     |  |
//     |  +--EDIT_CONFIG1  (0x0100)
//     |  +--EDIT_CONFIG2  (0x3E3D) 
//     |  +--EDIT_VIEW_P1  (0x7012)
//     |  |  |
//     |  |  +--TOP            (0x0001)
//     |  |  +--BOTTOM         (0x0002)
//     |  |  +--LEFT           (0x0003)
//     |  |  +--RIGHT          (0x0004)
//     |  |  +--FRONT          (0x0005) 
//     |  |  +--BACK           (0x0006)
//     |  |  +--USER           (0x0007)
//     |  |  +--CAMERA         (0xFFFF)
//     |  |  +--LIGHT          (0x0009)
//     |  |  +--DISABLED       (0x0010)  
//     |  |  +--BOGUS          (0x0011)
//     |  |
//     |  +--EDIT_VIEW_P2  (0x7011)
//     |  |  |
//     |  |  +--TOP            (0x0001)
//     |  |  +--BOTTOM         (0x0002)
//     |  |  +--LEFT           (0x0003)
//     |  |  +--RIGHT          (0x0004)
//     |  |  +--FRONT          (0x0005) 
//     |  |  +--BACK           (0x0006)
//     |  |  +--USER           (0x0007)
//     |  |  +--CAMERA         (0xFFFF)
//     |  |  +--LIGHT          (0x0009)
//     |  |  +--DISABLED       (0x0010)  
//     |  |  +--BOGUS          (0x0011)
//     |  |
//     |  +--EDIT_VIEW_P3  (0x7020)
//     |  +--EDIT_VIEW1    (0x7001) 
//     |  +--EDIT_BACKGR   (0x1200) 
//     |  +--EDIT_AMBIENT  (0x2100)
//     |  +--EDIT_OBJECT   (0x4000)
//     |  |  |
//     |  |  +--OBJ_TRIMESH   (0x4100)      
//     |  |  |  |
//     |  |  |  +--TRI_VERTEXL          (0x4110) 
//     |  |  |  +--TRI_VERTEXOPTIONS    (0x4111)
//     |  |  |  +--TRI_MAPPINGCOORS     (0x4140) 
//     |  |  |  +--TRI_MAPPINGSTANDARD  (0x4170)
//     |  |  |  +--TRI_FACEL1           (0x4120)
//     |  |  |  |  |
//     |  |  |  |  +--TRI_SMOOTH            (0x4150)   
//     |  |  |  |  +--TRI_MATERIAL          (0x4130)
//     |  |  |  |
//     |  |  |  +--TRI_LOCAL            (0x4160)
//     |  |  |  +--TRI_VISIBLE          (0x4165)
//     |  |  |
//     |  |  +--OBJ_LIGHT    (0x4600)
//     |  |  |  |
//     |  |  |  +--LIT_OFF              (0x4620)
//     |  |  |  +--LIT_SPOT             (0x4610) 
//     |  |  |  +--LIT_UNKNWN01         (0x465A) 
//     |  |  | 
//     |  |  +--OBJ_CAMERA   (0x4700)
//     |  |  |  |
//     |  |  |  +--CAM_UNKNWN01         (0x4710)
//     |  |  |  +--CAM_UNKNWN02         (0x4720)  
//     |  |  |
//     |  |  +--OBJ_UNKNWN01 (0x4710)
//     |  |  +--OBJ_UNKNWN02 (0x4720)
//     |  |
//     |  +--EDIT_UNKNW01  (0x1100)
//     |  +--EDIT_UNKNW02  (0x1201) 
//     |  +--EDIT_UNKNW03  (0x1300)
//     |  +--EDIT_UNKNW04  (0x1400)
//     |  +--EDIT_UNKNW05  (0x1420)
//     |  +--EDIT_UNKNW06  (0x1450)
//     |  +--EDIT_UNKNW07  (0x1500)
//     |  +--EDIT_UNKNW08  (0x2200)
//     |  +--EDIT_UNKNW09  (0x2201)
//     |  +--EDIT_UNKNW10  (0x2210)
//     |  +--EDIT_UNKNW11  (0x2300)
//     |  +--EDIT_UNKNW12  (0x2302)
//     |  +--EDIT_UNKNW13  (0x2000)
//     |  +--EDIT_UNKNW14  (0xAFFF)
//     |
//     +--KEYF3DS (0xB000)
//        |
//        +--KEYF_UNKNWN01 (0xB00A)
//        +--............. (0x7001) ( viewport, same as editor )
//        +--KEYF_FRAMES   (0xB008)
//        +--KEYF_UNKNWN02 (0xB009)
//        +--KEYF_OBJDES   (0xB002)
//           |
//           +--KEYF_OBJHIERARCH  (0xB010)
//           +--KEYF_OBJDUMMYNAME (0xB011)
//           +--KEYF_OBJUNKNWN01  (0xB013)
//           +--KEYF_OBJUNKNWN02  (0xB014)
//           +--KEYF_OBJUNKNWN03  (0xB015)  
//           +--KEYF_OBJPIVOT     (0xB020)  
//           +--KEYF_OBJUNKNWN04  (0xB021)  
//           +--KEYF_OBJUNKNWN05  (0xB022)  
//
// Once you know how to read chunks, all you have to know is the ID you are looking for
// and what data is stored after that ID.  You need to get the file format for that.
// I can give it to you if you want, or you can go to www.wosit.org for several versions.
// Because this is a proprietary format, it isn't a official document.
//
// I know there was a LOT of information blown over, but it is too much knowledge for
// one tutorial.  In the animation tutorial that I eventually will get to, some of
// the things explained here will be explained in more detail.
// 
// I would like to thank wosit.org and Terry Caton (tcaton@umr.edu) for his help on this.
//
// Let me know if this helps you out!
// 
// 
// Ben Humphrey (DigiBen)
// Game Programmer
// DigiBen@GameTutorials.com
// Co-Web Host of www.GameTutorials.com
//
//  2001 GameTutorials
