#ifndef _MAIN_H
#define _MAIN_H

#include <iostream>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <GL/gl.h>                                      // Header File For The OpenGL32 Library
#include <GL/glu.h>                                     // Header File For The GLu32 Library
#include <GL/glx.h>
#include <SDL/SDL_image.h>
#include <string>
#include <vector>

#include "SDL.h"

using namespace std;

typedef unsigned char BYTE;
typedef unsigned char byte;

extern int VideoFlags;          // The Pixel Format flags
extern SDL_Surface *MainWindow;  // Our Drawable portion of the window


#define SCREEN_WIDTH 800                                // We want our screen width 800 pixels
#define SCREEN_HEIGHT 600                               // We want our screen height 600 pixels
#define SCREEN_DEPTH 16                                 // We want 16 bits per pixel


#define MAX(a,b) (a>b) ? a : b
#define MIN(a,b) (a<b) ? a : b

//////////// *** NEW *** ////////// *** NEW *** ///////////// *** NEW *** ////////////////////

// This file includes all of the model structures that are needed to load
// in a .3DS file.  If you intend to do animation you will need to add on
// to this.  These structures only support the information that is needed
// to load the objects in the scene and their associative materials.

#define MAX_TEXTURES 100                                // The maximum amount of textures to load


typedef struct {
	int orient;
	float	raio, theta, phi;

} camera;

// This is our 3D point class.  This will be used to store the vertices of our model.
class CVector3 
{
public:
    float x, y, z;
};

// This is our 2D point class.  This will be used to store the UV coordinates.
class CVector2 
{
public:
    float x, y;
};

// This is our 2D fixed point class.
class Coord2 
{
public:
    int x, y;
};

// This is our face structure.  This is is used for indexing into the vertex 
// and texture coordinate arrays.  From this information we know which vertices
// from our vertex array go to which face, along with the correct texture coordinates.
class tFace
{
public:
    int vertIndex[3];           // indicies for the verts that make up this triangle
    int coordIndex[3];          // indicies for the tex coords to texture this face
};

// This holds the information for a material.  It may be a texture map of a color.
// Some of these are not used, but I left them because you will want to eventually
// read in the UV tile ratio and the UV tile offset for some models.
class tMaterialInfo
{
public:
    char  strName[255];         // The texture name
    char  strFile[255];         // The texture file name (If this is set it's a texture map)
    BYTE  color[3];             // The color of the object (R, G, B)
    int   texureId;             // the texture ID
    float uTile;                // u tiling of texture  (Currently not used)
    float vTile;                // v tiling of texture  (Currently not used)
    float uOffset;              // u offset of texture  (Currently not used)
    float vOffset;              // v offset of texture  (Currently not used)
} ;

// This holds all the information for our model/scene. 
// You should eventually turn into a robust class that 
// has loading/drawing/querying functions like:
// LoadModel(...); DrawObject(...); DrawModel(...); DestroyModel(...);
class t3DObject 
{
public:
    int  numOfVerts;            // The number of verts in the model
    int  numOfFaces;            // The number of faces in the model
    int  numTexVertex;          // The number of texture coordinates
    int  materialID;            // The texture ID to use, which is the index into our texture array
    bool bHasTexture;           // This is TRUE if there is a texture map for this object
    char strName[255];          // The name of the object
    CVector3  *pVerts;          // The object's vertices
    CVector3  *pNormals;        // The object's normals
    CVector2  *pTexVerts;       // The texture's UV coordinates
    tFace *pFaces;              // The faces information of the object
};

// This holds our model information.  This should also turn into a robust class.
// We use STL's (Standard Template Library) vector class to ease our link list burdens. :)

class t3DModel 
{
public:
    int numOfObjects;                   // The number of objects in the model
    int numOfMaterials;                 // The number of materials for the model
    unsigned int g_Texture[MAX_TEXTURES];
    t3DModel();
    ~t3DModel();
    vector<tMaterialInfo> pMaterials;   // The list of material information (Textures and colors)
    vector<t3DObject> pObject;          // The object list for our model
    void t3DModel::load (char *filename);
    void t3DModel::center ();
    void t3DModel::scale (float sx, float sy, float sz);
    void t3DModel::swap (char axis);
    void t3DModel::mirror (char axis);
private:
    CVector3  m_maxVerts, m_minVerts;
};

class tContext
{
public:
    int   g_ViewMode;
    bool  g_bLighting;                           // Turn lighting on initially
    float g_RotateX;                           // This is the current value at which the model is rotated
    float g_RotationSpeed;                           // This is the speed that our model rotates.  (-speed rotates left)
    CVector3 position;
    tContext(): g_ViewMode(GL_TRIANGLES), g_bLighting(true), g_RotateX(0.0f), g_RotationSpeed(5.0f) {};
    ~tContext() {};    
};

class tGrid 
{
  public:
  int ncols,nrows,selected;
  float dx,dy,wave_amp,roll;  
  bool animate,tag;
  
  GLuint texture[1]; 
  vector<CVector3> points;

  tGrid(): ncols(8), nrows(8) , dx(1.0) , dy(1.0), animate(false), wave_amp(0.0), roll(0.0), tag(false), selected(-2) {}; 
  ~tGrid() {};
  void tGrid::tessel();
  void tGrid::loadTex(char *filename);
  void tGrid::draw();
  void tGrid::size(float width, float height) {dx = width/(float)ncols; dy = height/(float)nrows;}
  void tGrid::grid(int _ncols, int _nrows) {ncols = _ncols; nrows = _nrows;}
  void tGrid::set_amplitude(float amp) {wave_amp = amp;}
  void tGrid::toggle_animation() {animate = !animate;}
  void tGrid::processHits (GLint hits, GLuint buffer[]);
  void tGrid::pick(int button, int state, int x, int y);
};

class tActor
{
public:
  t3DModel model;
  tContext cxt;
  tActor();
  tActor(char *filename, CVector3& scale);
  ~tActor();
  void tActor::moveto(tGrid* grid, float col, float row);
  void tActor::flip();
  void tActor::render(bool isday);
  void tActor::tActor::set_size (int xsize, int ysize) {sizex = xsize;sizey = ysize;};
  void tActor::tActor::set_position (int x, int y) {posx = x;posy = y;};

  int posx,posy;
  int sizex,sizey;
};

class tFont
{
  public:
  GLuint  base;
  tFont();
  ~tFont();
  void tFont::print (const char *fmt, ...);
  void tFont::position(float posx, float posy);
};

class tScene
{
  public:
  bool isday;
  vector<tActor*> actor;
  GLuint texture[5];   
  GLuint texture_mask[5];   
  tGrid grid;
  tGrid water;
  tFont text;
  int table[8][8];
  int frame[8][8];
  CVector3 campos;
  tScene(): isday(true) {};
  ~tScene() {};
  void tScene::render();
  void tScene::explosion(int col, int row);
  void tScene::place_ships();
};


//////////// *** NEW *** ////////// *** NEW *** ///////////// *** NEW *** ////////////////////
// This allows us to input ship positions
int Play(void *scene);

// This draws our scene
int DrawScene(void *scene);

// This Controls our MainLoop
int MainLoop(void* scn);

// This toggles between the Full Screen mode and the windowed mode
void ToggleFullScreen(void);

void CreateTexture(unsigned int textureArray[],char *strFileName,int textureID);
void CreateMaskedTexture(unsigned int textureArray[],char *strFileName, char *strMaskName, int textureID);

// This allows us to configure our window for OpenGL and backbuffered
void SetupPixelFormat(void);

// This is our own function that makes creating a window modular and easy
void CreateMyWindow(const char *strWindowName, int width, int height, int VideoFlags);

// This inits our screen translations and projections
void SizeOpenGLScreen(int width, int height);

// This sets up OpenGL
void InitializeGL(int width, int height);

// Build Grid Points
bool CreateGrid(vector<CVector3*> points, int ncols, int nrows);

// Load Grid Textures
bool LoadGridTextures( );

// Draw Grid Points
bool DrawGrid( vector<CVector3*> points, int ncols, int nrows);

// This initializes the whole program
void Init();

bool isExtensionSupported(string ext);

// This handles the keypress events generated when the user presses a key
void HandleKeyPressEvent(SDL_keysym * keysym, tContext& g_cxt);

// This shuts down SDL and quits program
void Quit(int ret_val);


#endif  // _MAIN_H_

/////////////////////////////////////////////////////////////////////////////////
////
//// * QUICK NOTES *
////
//// This file includes all the structures that you need to hold the model data.
//// Of course, if you want a robust loader, you need some more things for animation, etc..
//// If you intend to use this code, I would make the model and object structures classes.
//// This way you can have a bunch of helper functions like Import(), Translate(), Render()...
////
//// * What's An STL (Standard Template Library) Vector? *
//// Let me quickly explain the STL vector for those of you who are not familiar with them.
//// To use a vector you must include <vector> and use the std namespace: using namespace std;
//// A vector is an array based link list.  It allows you to dynamically add and remove nodes.
//// This is a template class so it can be a list of ANY type.  To create a vector of type
//// "int" you would say:  vector<int> myIntList;
//// Now you can add a integer to the dynamic array by saying: myIntList.push_back(10);
//// or you can say:  myIntList.push_back(num);.  The more you push back, the larger
//// your array gets.  You can index the vector like an array.  myIntList[0] = 0;
//// To get rid of a node you use the pop_back() function.  To clear the vector use clear().
//// It frees itself so you don't need to worry about it, except if you have data
//// structures that need information freed from inside them, like our objects.
////
//// The Import() function would decide which loader it needed to call, depending on the extension.
////
////
////
//// Ben Humphrey (DigiBen)
//// Game Programmer
//// DigiBen@GameTutorials.com
//// Co-Web Host of www.GameTutorials.com
////
////
////
