#ifndef __GAMECORE__
#define __GAMECORE__

#include <windows.h>			/* must include this before GL/gl.h */
#include <GL/gl.h>			/* OpenGL header file */
#include <GL/glu.h>			/* OpenGL utilities header file */
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <mmsystem.h>

#include <iostream>
#include <fstream>
#include <string>
#include <vector>
#include <cmath>
using namespace std;

#include "3ds.h"
#include "CAux.h"

#ifdef _GLUT_ENABLED
#include <GL/glut.h>
#endif

//Definicoes de modelo
#ifdef _DEBUG
#define MODEL_FILE1  "cena1.3ds"
#define MODEL_FILE2  "cenario0.3ds"
#else
#define MODEL_FILE1  "Resources\\cena1.3ds"
#define MODEL_FILE2  "Resources\\cenario0.3ds"
#endif

#define MDL_FILE1     0
#define MDL_FILE2     1

enum
{
  PLANE,
  BOMB1,
  BOMB2,
  BOMB3
};

//Definicoes de limites do aviao (eixo y invertido)
#define MIN_HORZ      44
#define MAX_HORZ      160
#define MAX_VERT      -60
#define MIN_VERT      -110

#define MAX_X         450
#define MIN_X         0

#define MIN_BOMB      15
#define MAX_BOMB      360

//Definicoes de velocidade
#define PLANE_SPEED   0.80f
#define BOMB_SPEED    0.5f

#define BOMB1_SPD     0.2f
#define BOMB2_SPD     0.5f
#define BOMB3_SPD     0.8f

#define COORD_OFFSET  2.0f

//Definicoes do alvo
#define TARGET_POSX0  150.0f
#define TARGET_POSY0  76.0f
#define TARGET_POSZ0  240.0f
#define TARGET_POSX1  190.0f
#define TARGET_POSY1  4.0f
#define TARGET_POSZ1  200.0f

//Dimensao da tela
#define WSIZE_X       800
#define WSIZE_Y       600

//Definicoes de visualizacao e zoom
#define DEFAULT       0   //Vista em perspectiva
#define UPVIEW        1   //Vista superior  X/Z
#define FRONTVIEW     2   //Vista traseira -Z/Y
#define LEFTVIEW      3   //Vista lateral   X/Y
#define ZOOMIN        4
#define ZOOMOUT       5
#define NONE          -1

#define ZOOMFACT      0.02
#define MINZOOM       -2.84f
#define MAXZOOM       1.27f

//Definicoes de iluminacao
#define NIGHT         0
#define DAY           1
#define NGL_LIGHTS    8 //Numero de luzes suportadas no OpenGL

//Definicoes do jogo
#define MAX_TRIES     5

//Definicoes de texturas
#define MAX_NUM_TEXTURES    20

//Estrutura de coordenadas 3D para movimentacao dos objetos
class CCoord3D 
{
public:
	float x, y, z;
  bool  bPress;

  CCoord3D ()
  {
    x = 0;
    y = 0;
    z = 0;
    bPress = false;
  }

};


//Definicao da classe principal
class CGameCore
{
public:
  CGameCore ();
  ~CGameCore ();

  //Transmite dados do teclado para o core do jogo
  void GetKbdData (int Key);

  //Verifica se o objeto foi instanciado com sucesso
  bool CheckStatus();

  //Laco principal
  int mainloop ();
  
  //Inicia ambiente OpenGL
  bool InitOpenGL();

  //Checa status de fim de jogo
  bool IsGameOver();

  //Laco da tela de encerramento
  void EndLoop ();

  //Exibe na tela dados do arquivo de modelos
  void ShowModelData ();

private:
  //Objeto de metodos auxiliares
  CAux *pAux;
  
  //Atualiza e limpa a tela
  void InitDisplay ();

  //Desenha o aviao
  void DrawPlane(float dx, float dy, float dz);

  //Desenha o projetil (retorna true) se acertou o alvo
  bool DrawBomb(float dx, float dy, float dz, int iSize = 0, int iIncl = 0);

  //Desenha um objeto
  void DrawObject (t3DObject *object);

  //Desenha o cenario
  void CreateScenario();

  //Retorna status de acerto
  bool GetHitStatus ();

  //Escreve um texto
  void OutText (CVector3 color, float PosX, float PosY, float PosZ, char *cText, ...);

  //Desenha o placar e informacoes ao jogador
  void DrawScore ();

  //Desenha cubo de teste
  void DrawCube (float dx, float dy, float dz);

#ifndef _GLUT_ENABLED
  //Cria ambiente para uso de fontes
  void BuildFont();

  //Destroi ambiente para uso de fontes
  void KillFont();
#endif

  //Define a janela de visualizacao
  void SetViewport (int iModeVar);

  //Flag da tecla ENTER
  bool  bEnter;
  
  //Tipo da bomba
  int   iTipoBomba;

//***********************************************************************
//* Animacao e modelamento
//***********************************************************************
  //Gera numeros aleatorios
  float rnd ();

  //Registra as coordenadas de acordo com entradas do teclado
  CCoord3D  coord;
  
  //Coordenadas do aviao
  CCoord3D  plcoord;

  //Coordenadas da bomba
  CCoord3D  bcoord;

  //Status da classe
  bool      bStatus;
  
  //Para uso da fonte
  int iBase;

  //Para uso do modo do viewport
  int iMode;

  //Registrador de periodo (dia/noite);
  int iRegPer;

  //Registrador de zoom
  float fZoomFact;

//***********************************************************************
//* Controle de acertos, etc
//***********************************************************************

  //Flag do acerto
  bool      bHit;

  //Contagem de tentativas
  int       iTry;

  //Flag de fim de jogo
  bool      bGameOver;

//***********************************************************************
//* Iluminacao e tonalizacao
//***********************************************************************
typedef struct _lights
{
  GLfloat LightAmbient [4];
  GLfloat LightDiffuse [4];
  GLfloat LightPosition[4];
  GLfloat LightDirection[4];

  _lights ()
  {
    memset (LightAmbient, 0, 4);
    memset (LightDiffuse, 0, 4);
    memset (LightPosition, 0, 4);
    memset (LightDirection, 0, 4);
  }
} GAME_LIGHTS;

  //Inicia estruturas de iluminacao e habilita tonalizacao
  void InitLighting (int iMode = NIGHT);
  
  //Muda iluminacao de acordo com o perfil
  void SelectAmbientLight (int iMode = NIGHT);
  
  GAME_LIGHTS LightItem[NGL_LIGHTS];

  //P/ Debug
  float TesteX;
  float TesteY;
  float TesteZ;

//***********************************************************************
//* Materiais
//***********************************************************************

  GLfloat fvAmbient[4];
  GLfloat fvDiffuse[4];
  GLfloat fvSpecular[4];
  GLfloat fvEmission[4];
  GLfloat fvNo_mat[4];
  GLfloat fShininess;

  //Seta cor do objeto e se emite ou nao luz
  void SetMaterial(CVector3 color, bool bSelf = false);

//***********************************************************************
//* Texturas
//***********************************************************************
  AUX_RGBImageRec *TextureImage[MAX_NUM_TEXTURES];
  GLuint		texture[MAX_NUM_TEXTURES];

  //Carrega as texturas do jogo
  int  LoadTextures ();

  //Libera memoria usada na carga das texturas
  void FlushTextures ();
  
  //Passa os dados da textura ao OpenGL
  void AssignTextures (int ID);

  enum
  {
    TEXTNULL,
    PREDIO1,
    PREDIO2,
    PREDIO3,
    PREDIO4,
    PREDIO5,
    PREDIO6,
    PREDIO7,
    PREDIO8,
    PREDIO9,
    PREDIO10,
    TERRENO,
    CIMENTO,
    ALVO,
    ALVO_N,
    JATO,
    MISSIL,
    GRAMA
  };
};


#endif
