#include "..\interface\ac3dReader.h"
#include "malloc.h"
#include "io.h"
#include "fcntl.h"
#include "string.h"
#include "stdio.h"
#include "stdlib.h"
#include "stdarg.h"
#include "limits.h"
#define VERBOSE 0
#define AC3D_DEBUG 0

//#define GOURAUD_SHADING 0

#define EMPTY_COLOR 0xffffffffff

typedef enum
{
	AC3DB = 0,
	MATERIAL,
	OBJECT,
	KIDS,
	NAME,
	DATA,
	CREASE,
	NUMVERT,
	NUMSURF,
	SURF,
	MAT,
	REFS,
	TEXTURE,
	NUM_OF_KEYWORDS
} ac3dfile_keyword_t;

typedef char ac3dfile_keyword[256];

static ac3dfile_keyword keywords[] = 
{
	"AC3Db",
	"MATERIAL",
	"OBJECT",
	"kids",
	"name",
	"data",
	"crease",
	"numvert",
	"numsurf",
	"SURF",
	"mat",
	"refs",
	"texture",
	"not found"
};

typedef enum{
	INITIALIZED=0,
	WORLD_OBJECT,
	COMMON_OBJECT,
	OBJECT_VERTEX,
	OBJECT_SURF,
	OBJECT_SURF_REFS,
} ac3dfile_parser_state_t;

typedef struct{
	ac3dfile_t* ac3dfile;
	ac3dfile_parser_state_t state;
	int currentObjectIndex;
	long int currentVertexIndex;
	long int currentSurfIndex;
	int currentSurfRefIndex;
	int currentMaterialIndex;
	ac3dfile_material_t* currentMaterial;
} ac3dfile_parser_t;


static void ac3dfile_parser_print(char* fmt, ...)
{
	va_list list;
	
	if(VERBOSE>0)
	{
		va_start(list,fmt);
		vprintf(fmt,list);
	}
}


static char* findNextEndOfLine(char* buffer, unsigned int size)
{
	unsigned int i = 0;
	
	while(*buffer!='\n' && i<size)
	{
		i++;
		buffer++;
	}

	if(i==size)
	{
		return NULL;
	}
	else
	{
		return buffer;
	}
}

static int openFile(char* filepath)
{
	int fd = open(filepath, _O_RDONLY, 0);
	return fd;
}

static int fillBuffer(int fd, char* buffer, char* currentPosition, unsigned int bufferSize)
{
	int readed;
	int toRead;
	if(currentPosition>buffer)
	{
		memmove(buffer,currentPosition,bufferSize-(currentPosition-buffer));	
	}
	
	toRead = AC3D_BUFFER_SIZE-(bufferSize-(currentPosition-buffer));
	//printf("To Read: %d\n",toRead);
	readed = read(fd, buffer-(currentPosition-buffer)+bufferSize, (unsigned int) (toRead));
	return bufferSize-(currentPosition-buffer) + readed;
}

static ac3dfile_keyword_t getAC3DKeyword(char* line)
{
	int i = 0;
	for(i=0;i<NUM_OF_KEYWORDS;i++)
	{
		if(strncmp(line,keywords[i],strlen(keywords[i]))==0)
		{
			break;
		}
	}

	return (ac3dfile_keyword_t)i;
}

static void storeMaterialInArray(ac3dfile_parser_t* pParser)
{
	int i=0;
	ac3dfile_material_t* aux = pParser->ac3dfile->materialLinkedList;

	pParser->ac3dfile->materials = MALL_ALLOC(sizeof(ac3dfile_material_t*)*(pParser->currentMaterialIndex+1));
	
	for(i=0;i<=pParser->currentMaterialIndex;i++)
	{
		pParser->ac3dfile->materials[i] = aux;

		if(aux->next!=NULL)
			aux = aux->next;
	}


}

static ac3dReturn_t parseTexture(ac3dfile_parser_t* pParser, char* line, char* endLine)
{
	char* aux;
	char* aux2;
	pParser->ac3dfile->objects[pParser->currentObjectIndex]->texture = 1;

	aux = line + 9;

	aux2 = strchr(aux,'\"');

	pParser->ac3dfile->objects[pParser->currentObjectIndex]->textureFile = MALL_ALLOC(sizeof(char)*(aux2-aux+1));

	strncpy(pParser->ac3dfile->objects[pParser->currentObjectIndex]->textureFile,aux,aux2-aux);

}

static ac3dReturn_t parseMaterial(ac3dfile_parser_t* pParser, char* line, char* endLine)
{
	char* aux;
	char* aux2;
	ac3dReturn_t result = AC3D_SUCCESS;
	ac3dfile_material_t* material;

	pParser->currentMaterialIndex++;

	material = MALL_ALLOC(sizeof(ac3dfile_material_t));

	material->next = NULL;

	aux = strchr(line,'\"');

	if(aux==NULL || aux>endLine)
	{
		result = AC3D_FAILED;
	}

	if(result == AC3D_SUCCESS)
	{
		aux2 = strchr(aux+1,'\"');

		if(aux2==NULL || aux2>endLine)
		{
			result = AC3D_FAILED;
		}

		if(result == AC3D_SUCCESS)
		{
			material->name = MALL_ALLOC(aux2-aux);
			strncpy(material->name,aux,aux2-aux);
			material->name[aux2-aux]='\0';
		}
	}

	if(result == AC3D_SUCCESS)
	{
		aux = aux2 + 6;
		material->r = atof(aux);
		material->diffuse[0] = material->r;
		
		aux = strchr(aux,' ');

		//todo error check
		material->g = atof(++aux);
		material->diffuse[1] = material->g;
		
		aux = strchr(aux,' ');

		material->b = atof(++aux);
		material->diffuse[2] = material->b;

		aux = strstr(aux,"amb");

		if(aux==NULL || aux>endLine)
		{
			result = AC3D_FAILED;
		}
	}

	if(result == AC3D_SUCCESS)
	{
		aux += 4;

		material->amb[0] = atof(aux);

		aux = strchr(aux,' ');

		//todo error check
		material->amb[1] = atof(++aux);

		aux = strchr(aux,' ');

		material->amb[2] = atof(++aux);

		aux = strstr(aux,"emis");

		if(aux==NULL || aux>endLine)
		{
			result = AC3D_FAILED;
		}

	}

	if(result == AC3D_SUCCESS)
	{
		aux += 5;

		material->emis[0] = atof(aux);

		aux = strchr(aux,' ');

		//todo error check
		material->emis[1] = atof(++aux);

		aux = strchr(aux,' ');

		material->emis[2] = atof(++aux);

		aux = strstr(aux,"spec");

		if(aux==NULL || aux>endLine)
		{
			result = AC3D_FAILED;
		}

	}

	if(result == AC3D_SUCCESS)
	{
		aux += 5;

		material->spec[0] = atof(aux);

		aux = strchr(aux,' ');

		//todo error check
		material->spec[1] = atof(++aux);

		aux = strchr(aux,' ');

		material->spec[2] = atof(++aux);

		aux = strstr(aux,"shi");

		if(aux==NULL || aux>endLine)
		{
			result = AC3D_FAILED;
		}

	}

	if(result == AC3D_SUCCESS)
	{
		aux += 4;

		material->shi = atoi(aux);

		aux = strstr(aux,"trans");

		if(aux==NULL || aux>endLine)
		{
			result = AC3D_FAILED;
		}

	}

	if(result == AC3D_SUCCESS)
	{
		aux += 6;

		material->trans = atoi(aux);

		if(pParser->currentMaterial!=NULL)
		{
			pParser->currentMaterial->next = material;
			pParser->currentMaterial = material;
		}
		else
		{
			pParser->ac3dfile->materialLinkedList = material;
			pParser->currentMaterial = material;
		}

		pParser->ac3dfile->numberOfMaterials++;

		//printf("MATERIAL \"%s\" rgb %f %f %f",material->name,material->r,material->g,material->b);

	}




}

static ac3dReturn_t parseObject(ac3dfile_parser_t* pParser, char* line)
{
	char* objectType;

	objectType = line + strlen("Object") + 1;

	if(strncmp(objectType,"world",strlen("world"))==0)
	{
		pParser->state = WORLD_OBJECT;
		ac3dfile_parser_print("World Object\n"); 

		//Store Material
		storeMaterialInArray(pParser);

	}
	else
	{
		pParser->state = OBJECT;
		//@todo
		//Alloc object
		pParser->currentObjectIndex++;
		pParser->ac3dfile->objects[pParser->currentObjectIndex]=MALL_ALLOC(1*sizeof(ac3dfile_object_t));
		ac3dfile_parser_print("Common Object %d\n",pParser->currentObjectIndex);
	}

	return AC3D_SUCCESS;
}

static ac3dReturn_t parseKids(ac3dfile_parser_t* pParser, char* line)
{
	//Get number Of Kids
	unsigned int numberOfKids;

	numberOfKids = atoi(line+strlen("kids")+1);

	if(pParser->state == WORLD_OBJECT)
	{
		//@todo
		//Alloc array of obects
		pParser->ac3dfile->numberOfOjects = numberOfKids;
		pParser->ac3dfile->objects = (ac3dfile_object_t**)MALL_ALLOC(numberOfKids*sizeof(ac3dfile_object_t*));
	}

	ac3dfile_parser_print("Number of Kids: %d\n", numberOfKids);

	return AC3D_SUCCESS;
}

static ac3dReturn_t parseName(ac3dfile_parser_t* pParser, char* line, char* endLine)
{
	char* name;
	char* aux = line+strlen("name")+2;

	name = MALL_ALLOC((endLine-aux+1)*sizeof(char));
	strncpy(name,aux,endLine-1-aux);

	name[endLine-1-aux]='\0';
	ac3dfile_parser_print("Object %d name: %s\n",pParser->currentObjectIndex,name);

	//@todo Store Name
	pParser->ac3dfile->objects[pParser->currentObjectIndex]->name = name;

	return AC3D_SUCCESS;

}

static ac3dReturn_t parseNumVert(ac3dfile_parser_t* pParser, char* line)
{
	unsigned long int numVert;

	numVert = atol(line+strlen("numvert")+1);

	pParser->ac3dfile->objects[pParser->currentObjectIndex]->numberOfVertices = numVert;
	pParser->ac3dfile->objects[pParser->currentObjectIndex]->vertices = MALL_ALLOC(numVert*3*sizeof(float));

	pParser->ac3dfile->objects[pParser->currentObjectIndex]->verticeColors = MALL_ALLOC(numVert*3*sizeof(float));
	memset(pParser->ac3dfile->objects[pParser->currentObjectIndex]->verticeColors,0xffff,numVert*3*sizeof(float));

	pParser->currentVertexIndex = -1;

	pParser->ac3dfile->objects[pParser->currentObjectIndex]->textCoordArray = MALL_ALLOC(sizeof(float)*2*numVert);

	//@todo alloc vertices

	pParser->state = OBJECT_VERTEX;

	ac3dfile_parser_print("Object %d - Number Of Vertices: %d\n",pParser->currentObjectIndex,numVert);

	return AC3D_SUCCESS;

}

static ac3dReturn_t parseNumSurf(ac3dfile_parser_t* pParser, char* line)
{
	unsigned long int numSurf;

	numSurf = atol(line+strlen("numsurf")+1);

	pParser->ac3dfile->objects[pParser->currentObjectIndex]->numberOfSurfaces = numSurf;
	pParser->ac3dfile->objects[pParser->currentObjectIndex]->surfaces = MALL_ALLOC(numSurf*sizeof(ac3dfile_surf_t*));
	pParser->currentSurfIndex=-1;

	//@todo alloc surfaces

	pParser->state = OBJECT_SURF;

	ac3dfile_parser_print("Object %d - Number Of Surfaces: %d\n",pParser->currentObjectIndex,numSurf);

	return AC3D_SUCCESS;
}

static ac3dReturn_t parseSurf(ac3dfile_parser_t* pParser, char* line)
{
	//@todo increase currentSurf
	pParser->currentSurfIndex++;
	pParser->ac3dfile->objects[pParser->currentObjectIndex]->surfaces[pParser->currentSurfIndex] = MALL_ALLOC(sizeof(ac3dfile_surf_t));
	pParser->state = OBJECT_SURF;
	return AC3D_SUCCESS;
}

static ac3dReturn_t parseMat(ac3dfile_parser_t* pParser, char* line)
{
	unsigned int mat;

	mat = atoi(line+strlen("mat")+1);

	pParser->ac3dfile->objects[pParser->currentObjectIndex]->surfaces[pParser->currentSurfIndex]->material = pParser->ac3dfile->materials[mat];

}

static ac3dReturn_t parseRefs(ac3dfile_parser_t* pParser, char* line)
{
	unsigned int numRefs;

	numRefs = atoi(line+strlen("refs")+1);

	pParser->state = OBJECT_SURF_REFS;

	//@todo numRefs
	//Allocate 
	pParser->ac3dfile->objects[pParser->currentObjectIndex]->surfaces[pParser->currentSurfIndex]->numberOfRefs = numRefs;
	pParser->ac3dfile->objects[pParser->currentObjectIndex]->surfaces[pParser->currentSurfIndex]->references = MALL_ALLOC(sizeof(long int)*numRefs);
	pParser->currentSurfRefIndex=-1;

	ac3dfile_parser_print("Surface %d - Number Of Refs: %d\n",pParser->currentSurfIndex,numRefs);

	return AC3D_SUCCESS;

}

static ac3dReturn_t parseVertex(ac3dfile_parser_t* pParser, char* line)
{
	float vertex[3];
	char* aux;
	ac3dReturn_t result = AC3D_SUCCESS;

	pParser->currentVertexIndex++;

	pParser->ac3dfile->objects[pParser->currentObjectIndex]->vertices[pParser->currentVertexIndex] = atof(line);

	aux = strchr(line,' ');

	if(aux==NULL)
	{
		result = AC3D_FAILED;
	}

	if(result==AC3D_SUCCESS)
	{
		pParser->currentVertexIndex++;
		pParser->ac3dfile->objects[pParser->currentObjectIndex]->vertices[pParser->currentVertexIndex] = atof(++aux);
	}

	aux = strchr(aux,' ');

	if(aux==NULL)
	{
		result = AC3D_FAILED;
	}

	if(result==AC3D_SUCCESS)
	{
		pParser->currentVertexIndex++;
		pParser->ac3dfile->objects[pParser->currentObjectIndex]->vertices[pParser->currentVertexIndex] = atof(++aux);
		ac3dfile_parser_print("Vertex %d(%d %d %d): %f %f %f\n",
			pParser->currentVertexIndex/3,
			pParser->currentVertexIndex-2,
			pParser->currentVertexIndex-1,
			pParser->currentVertexIndex,
			pParser->ac3dfile->objects[pParser->currentObjectIndex]->vertices[pParser->currentVertexIndex-2],
			pParser->ac3dfile->objects[pParser->currentObjectIndex]->vertices[pParser->currentVertexIndex-1],
			pParser->ac3dfile->objects[pParser->currentObjectIndex]->vertices[pParser->currentVertexIndex]);
	}

	//@todo
	//Store vertex
	//Increase currentVertex

	return result;
}

static ac3dReturn_t parseRef(ac3dfile_parser_t* pParser, char* line)
{
	unsigned long int ref;
	char* aux;
	float tcoord1;
	float tcoord2;

	aux = strchr(line,' ') +1;
	ref = atol(line);
	tcoord1 = atof(aux);

	aux = strchr(aux,' ') + 1;

	tcoord2 = atof(aux);


	//@todo Add to references[i]
	//Increase currentSurfRefIndex
	pParser->currentSurfRefIndex++;
	pParser->ac3dfile->objects[pParser->currentObjectIndex]->surfaces[pParser->currentSurfIndex]->references[pParser->currentSurfRefIndex]=ref;

	if(pParser->ac3dfile->objects[pParser->currentObjectIndex]->texture==1)
	{
		pParser->ac3dfile->objects[pParser->currentObjectIndex]->textCoordArray[ref*2]=tcoord1;
		pParser->ac3dfile->objects[pParser->currentObjectIndex]->textCoordArray[ref*2+1]=tcoord2;
	}

	ac3dfile_parser_print("Surface %d - Ref %d: %d\n",pParser->currentSurfIndex,pParser->currentSurfRefIndex,ref);

#ifdef GOURAUD_SHADING
	if(pParser->currentSurfRefIndex+1==pParser->ac3dfile->objects[pParser->currentObjectIndex]->surfaces[pParser->currentSurfIndex]->numberOfRefs)
	{
		int i=0;
		for(i=0;i<pParser->ac3dfile->objects[pParser->currentObjectIndex]->surfaces[pParser->currentSurfIndex]->numberOfRefs;i++)
		{
			if((long)pParser->ac3dfile->objects[pParser->currentObjectIndex]->verticeColors[
				pParser->ac3dfile->objects[pParser->currentObjectIndex]->surfaces[pParser->currentSurfIndex]->references[i]*3]==(long)LONG_MIN)
			{
				pParser->ac3dfile->objects[pParser->currentObjectIndex]->verticeColors[
					pParser->ac3dfile->objects[pParser->currentObjectIndex]->surfaces[pParser->currentSurfIndex]->references[i]*3]
				= pParser->ac3dfile->objects[pParser->currentObjectIndex]->surfaces[pParser->currentSurfIndex]->material->r;
				pParser->ac3dfile->objects[pParser->currentObjectIndex]->verticeColors[
					pParser->ac3dfile->objects[pParser->currentObjectIndex]->surfaces[pParser->currentSurfIndex]->references[i]*3+1]
				= pParser->ac3dfile->objects[pParser->currentObjectIndex]->surfaces[pParser->currentSurfIndex]->material->g;
				pParser->ac3dfile->objects[pParser->currentObjectIndex]->verticeColors[
					pParser->ac3dfile->objects[pParser->currentObjectIndex]->surfaces[pParser->currentSurfIndex]->references[i]*3+2]
				= pParser->ac3dfile->objects[pParser->currentObjectIndex]->surfaces[pParser->currentSurfIndex]->material->b;

			}
			else
			{
				pParser->ac3dfile->objects[pParser->currentObjectIndex]->verticeColors[
					pParser->ac3dfile->objects[pParser->currentObjectIndex]->surfaces[pParser->currentSurfIndex]->references[i]*3]
				=(pParser->ac3dfile->objects[pParser->currentObjectIndex]->surfaces[pParser->currentSurfIndex]->material->r +
					pParser->ac3dfile->objects[pParser->currentObjectIndex]->verticeColors[
					pParser->ac3dfile->objects[pParser->currentObjectIndex]->surfaces[pParser->currentSurfIndex]->references[i]*3])/2;
				pParser->ac3dfile->objects[pParser->currentObjectIndex]->verticeColors[
					pParser->ac3dfile->objects[pParser->currentObjectIndex]->surfaces[pParser->currentSurfIndex]->references[i]*3+1]
				= (pParser->ac3dfile->objects[pParser->currentObjectIndex]->surfaces[pParser->currentSurfIndex]->material->g +
					pParser->ac3dfile->objects[pParser->currentObjectIndex]->verticeColors[
					pParser->ac3dfile->objects[pParser->currentObjectIndex]->surfaces[pParser->currentSurfIndex]->references[i]*3+1])/2;

				pParser->ac3dfile->objects[pParser->currentObjectIndex]->verticeColors[
					pParser->ac3dfile->objects[pParser->currentObjectIndex]->surfaces[pParser->currentSurfIndex]->references[i]*3+2]
				= (pParser->ac3dfile->objects[pParser->currentObjectIndex]->surfaces[pParser->currentSurfIndex]->material->b +
					pParser->ac3dfile->objects[pParser->currentObjectIndex]->verticeColors[
					pParser->ac3dfile->objects[pParser->currentObjectIndex]->surfaces[pParser->currentSurfIndex]->references[i]*3+2])/2;				
			}
		}
	}
#endif
	
	return AC3D_SUCCESS;
}



ac3dReturn_t ac3dReader_Load(char* filepath, ac3dfile_t** ppAC3DFile)
{
	int fd;
	char buffer[AC3D_BUFFER_SIZE];
	ac3dReturn_t result = AC3D_SUCCESS;

	ac3dfile_t* ac3dfile;
	ac3dfile = (ac3dfile_t*)MALL_ALLOC(sizeof(ac3dfile_t));

	if(ac3dfile != NULL)
	{
		*ppAC3DFile = ac3dfile;
		fd = openFile(filepath);

		if(fd!=-1)
		{
			unsigned int bufferSize=0;
			char* beginLine;
			char* endLine;
			ac3dfile_parser_t fileParser;

			fileParser.state = INITIALIZED;
			fileParser.currentObjectIndex = fileParser.currentSurfIndex = fileParser.currentVertexIndex = fileParser.currentMaterialIndex = -1;
			fileParser.currentMaterial=NULL;
			fileParser.ac3dfile = ac3dfile;

			ac3dfile->numberOfMaterials=0;

			ac3dfile->filepath = MALL_ALLOC(sizeof(char)*(strlen(filepath)+1));
			strcpy(ac3dfile->filepath,filepath);

			ac3dfile_parser_print("File %s Opened\n",ac3dfile->filepath);

			beginLine = buffer;

			
			do
			{
				bufferSize = fillBuffer(fd, buffer, beginLine, bufferSize);

				//printf("Buffer Size: %d\n", bufferSize);
				beginLine = buffer;
				endLine = findNextEndOfLine(beginLine, bufferSize);
				
				if(endLine!=NULL)
				{
					//parse
					switch(getAC3DKeyword(beginLine))
					{
					case MATERIAL:
						parseMaterial(&fileParser,beginLine,endLine);
						break;
					case OBJECT:
						parseObject(&fileParser,beginLine);
						break;
					case KIDS:
						parseKids(&fileParser,beginLine);
						break;
					case NAME:
						parseName(&fileParser,beginLine,endLine);
						break;
					case DATA:
						break;
					case CREASE:
						break;
					case NUMVERT:
						parseNumVert(&fileParser, beginLine);
						break;
					case NUMSURF:
						parseNumSurf(&fileParser, beginLine);
						break;
					case SURF:
						parseSurf(&fileParser, beginLine);
						break;
					case MAT:
						parseMat(&fileParser, beginLine);
						break;
					case REFS:
						parseRefs(&fileParser,beginLine);
						break;
					case TEXTURE:
						parseTexture(&fileParser,beginLine,endLine);
						break;
					default:
						{
							if(fileParser.state==OBJECT_VERTEX)
							{
								//Parse Vertex
								parseVertex(&fileParser,beginLine);
							}
							else if(fileParser.state == OBJECT_SURF_REFS)
							{
								//Parse Surf Refs
								parseRef(&fileParser,beginLine);
							}
						}
						break;
					}

					//*endLine = '\0';
					//printf("%s\n", beginLine);
				} 
				else
				{
					//error
					//printf("error");
				}

				if(bufferSize>endLine-buffer)
					beginLine = endLine + 1;
				else
				{
					ac3dfile_parser_print("failed");
				}
				
			}while(bufferSize>endLine-buffer && result == AC3D_SUCCESS);

		}
		else
		{
			result = AC3D_SUCCESS;		
		}

	}

	return result;
	
}

void ac3dReader_Free(ac3dfile_t* ac3dfile)
{
	unsigned int i = 0;

	FREE(ac3dfile->filepath);

	for(i=0;i<ac3dfile->numberOfOjects;i++)
	{
		unsigned long int j = 0;

		FREE(ac3dfile->objects[i]->vertices);
		FREE(ac3dfile->objects[i]->name);

		for(j=0;j<ac3dfile->objects[i]->numberOfSurfaces;j++)
		{
			FREE(ac3dfile->objects[i]->surfaces[j]->references);
			FREE(ac3dfile->objects[i]->surfaces[j]);
		}

		FREE(ac3dfile->objects[i]->surfaces);
		FREE(ac3dfile->objects[i]);		
	}

	FREE(ac3dfile->objects);
	FREE(ac3dfile);

#if AC3D_DEBUG 1
	{
		printf("mem %d\n" ,mem);
	}
#endif

	return AC3D_SUCCESS;
}

void ac3dReader_Print(ac3dfile_t* ac3dfile)
{
	unsigned int i = 0;

	//printf("File: %s\n",ac3dfile->filepath);
    
	printf("AC3Db\n");
	//Printf default Material
	printf("MATERIAL \"DefaultWhite\" rgb 1 1 1  amb 1 1 1  emis 0 0 0  spec 0.5 0.5 0.5  shi 64  trans 0\n");
	printf("OBJECT world\nkids %d\n",ac3dfile->numberOfOjects);
	


	for(i=0;i<ac3dfile->numberOfOjects;i++)
	{
		unsigned long int j = 0;

		printf("OBJECT poly\n");
		printf("name \"%s\"\n",ac3dfile->objects[i]->name);
		printf("numvert %d\n",ac3dfile->objects[i]->numberOfVertices);

		for(j=0;j<ac3dfile->objects[i]->numberOfVertices;j++)
		{
			printf("%f %f %f\n", ac3dfile->objects[i]->vertices[j*3],ac3dfile->objects[i]->vertices[j*3+1],ac3dfile->objects[i]->vertices[j*3+2]);
		}

		printf("numsurf %d\n",ac3dfile->objects[i]->numberOfSurfaces);

		for(j=0;j<ac3dfile->objects[i]->numberOfSurfaces;j++)
		{
			unsigned int k=0;
			//Material set to default
			printf("SURF 0x20\nmat 0\nrefs %d\n",ac3dfile->objects[i]->surfaces[j]->numberOfRefs);

			for(k=0;k<ac3dfile->objects[i]->surfaces[j]->numberOfRefs;k++)
			{
				printf("%d 0 0\n",ac3dfile->objects[i]->surfaces[j]->references[k]);
			}

		}

		printf("kids 0\n");

	}

}