/*!
   @author Wu Shin-Ting - FEEC/Unicamp
   @date March/2016
   @see http://www.opengl-tutorial.org/beginners-tutorials/
 */

#include <iostream>

#ifdef WIN32
#define GLEW_STATIC
#include <GL/glew.h>
#endif
#define FREEGLUT_STATIC
#include <GL/freeglut.h>

#include <math.h>
#include <glm/glm.hpp> 
#include <glm/gtc/matrix_transform.hpp> 
#include <glm/gtc/type_ptr.hpp>  //! memory layout access
#include "LoadShaders.h"
#include <stdio.h>

#ifndef WIN32
#include <time.h>
#endif 

//! Function scope: std 
using namespace std;

//! Function declaration (prototype)
void Initialize(const char * title);
void Display(void);
void Finalize(void);
void Reshape(int width, int height);

#define slices 10
#define stacks 20

float aspect;
GLuint cube_prog;

GLuint vao;
GLuint cube_vbo;
GLuint vertexbuffer;
GLuint colorbuffer;

GLint mv_mat_loc;
GLint prj_mat_loc;

#ifndef WIN32
timespec start_tick;
#endif 

void Initialize()
{
  //! Create the program for rendering the scene from the light's POV. 
  glCreateProgram();

  ShaderInfo cube_info[] =
    {
      { GL_VERTEX_SHADER, "shaders/hello_cube.vs.glsl" },
      { GL_FRAGMENT_SHADER, "shaders/hello_cube.fs.glsl" },
      { GL_NONE, NULL }
    };

  cube_prog = LoadShaders(cube_info);

  mv_mat_loc = glGetUniformLocation(cube_prog, "MVMatrix");
  prj_mat_loc = glGetUniformLocation(cube_prog, "PMatrix");

  static const GLfloat g_vertex_buffer_data[] = {
    -1.0f, 1.0f, 1.0f, //! triangle 1 : begin
    -1.0f,-1.0f, 1.0f,
    -1.0f,-1.0f,-1.0f, //! triangle 1 : end
    -1.0f, 1.0f,-1.0f, //! triangle 2 : begin
    1.0f, 1.0f,-1.0f, 
    -1.0f,-1.0f,-1.0f, //! triangle 2 : end
    -1.0f,-1.0f,-1.0f,
    1.0f,-1.0f, 1.0f,
    1.0f,-1.0f,-1.0f, //! triangle 3 : end
    1.0f,-1.0f,-1.0f,
    1.0f, 1.0f,-1.0f,
    -1.0f,-1.0f,-1.0f, //! triangle 4 : end
    -1.0f,-1.0f,-1.0f,
    -1.0f, 1.0f,-1.0f,
    -1.0f, 1.0f, 1.0f, //! triangle 5 : end
    -1.0f,-1.0f, 1.0f,
    1.0f,-1.0f, 1.0f,
    -1.0f,-1.0f,-1.0f, //! triangle 6 : end
    -1.0f,-1.0f, 1.0f,
    -1.0f, 1.0f, 1.0f,
    1.0f,-1.0f, 1.0f, //! triangle 7 : end
    1.0f,-1.0f,-1.0f,
    1.0f, 1.0f, 1.0f,
    1.0f, 1.0f,-1.0f, //! triangle 8 : end
    1.0f, 1.0f, 1.0f,
    1.0f,-1.0f,-1.0f,
    1.0f,-1.0f, 1.0f, //! triangle 9 : end
    1.0f, 1.0f,-1.0f,
    1.0f, 1.0f, 1.0f,
    -1.0f, 1.0f,-1.0f, //! triangle 10 : end
    -1.0f, 1.0f,-1.0f,
    1.0f, 1.0f, 1.0f,
    -1.0f, 1.0f, 1.0f, //! triangle 11 : end
    -1.0f, 1.0f, 1.0f,
    1.0f, 1.0f, 1.0f,
    1.0f,-1.0f, 1.0f //! triangle 12 : end
  };

  //! One color for each vertex. They were generated randomly.
  static const GLfloat g_color_buffer_data[] = {
    0.583f,  0.771f,  0.014f,
    0.609f,  0.115f,  0.436f,
    0.327f,  0.483f,  0.844f,
    0.822f,  0.569f,  0.201f,
    0.435f,  0.602f,  0.223f,
    0.310f,  0.747f,  0.185f,
    0.597f,  0.770f,  0.761f,
    0.559f,  0.436f,  0.730f,
    0.359f,  0.583f,  0.152f,
    0.483f,  0.596f,  0.789f,
    0.559f,  0.861f,  0.639f,
    0.195f,  0.548f,  0.859f,
    0.014f,  0.184f,  0.576f,
    0.771f,  0.328f,  0.970f,
    0.406f,  0.615f,  0.116f,
    0.676f,  0.977f,  0.133f,
    0.971f,  0.572f,  0.833f,
    0.140f,  0.616f,  0.489f,
    0.997f,  0.513f,  0.064f,
    0.945f,  0.719f,  0.592f,
    0.543f,  0.021f,  0.978f,
    0.279f,  0.317f,  0.505f,
    0.167f,  0.620f,  0.077f,
    0.347f,  0.857f,  0.137f,
    0.055f,  0.953f,  0.042f,
    0.714f,  0.505f,  0.345f,
    0.783f,  0.290f,  0.734f,
    0.722f,  0.645f,  0.174f,
    0.302f,  0.455f,  0.848f,
    0.225f,  0.587f,  0.040f,
    0.517f,  0.713f,  0.338f,
    0.053f,  0.959f,  0.120f,
    0.393f,  0.621f,  0.362f,
    0.673f,  0.211f,  0.457f,
    0.820f,  0.883f,  0.371f,
    0.982f,  0.099f,  0.879f
  };

  char buf[1024];
  glGetProgramInfoLog(cube_prog, 1024, NULL, buf);
#ifdef _DEBUG
  std::cout << "Program Info Log: " << buf << std::endl;
#endif

  // Create a vertex array object
  glGenVertexArrays(1, &vao);
  glBindVertexArray(vao);
  glGenBuffers(1, &vertexbuffer);
  glBindBuffer(GL_ARRAY_BUFFER, vertexbuffer);
  glBufferData(GL_ARRAY_BUFFER, sizeof(g_vertex_buffer_data), g_vertex_buffer_data, GL_STATIC_DRAW);
  // Tell OpenGL the format of your array data
  glVertexAttribPointer(
			0,                                // attribute. No particular reason for 0, but must match the layout in the shader.
			3,                                // size
			GL_FLOAT,                         // type
			GL_FALSE,                         // normalized?
			0,                                // stride
			(void*)0                          // array buffer offset
			);

  glGenBuffers(1, &colorbuffer);
  glBindBuffer(GL_ARRAY_BUFFER, colorbuffer);
  glBufferData(GL_ARRAY_BUFFER, sizeof(g_color_buffer_data), g_color_buffer_data, GL_STATIC_DRAW);
  // Tell OpenGL the format of your array data
  glVertexAttribPointer(
			1,                                // attribute. No particular reason for 1, but must match the layout in the shader.
			3,                                // size
			GL_FLOAT,                         // type
			GL_FALSE,                         // normalized?
			0,                                // stride
			(void*)0                          // array buffer offset
			);

#ifndef WIN32
  clock_gettime(CLOCK_PROCESS_CPUTIME_ID, &start_tick);
#endif

  glClearColor(0.0f, 0.25f, 0.3f, 1.0f);
  glClearDepth(1.0f);
  glEnable(GL_DEPTH_TEST);
  glDepthFunc(GL_LEQUAL);
  glDisable(GL_CULL_FACE);
}

void Display(void)
{
#ifdef WIN32
  static const unsigned int start_time = GetTickCount();
  float t = float((GetTickCount() - start_time)) / float(0x3FFF);
#else
  timespec tick;
  clock_gettime(CLOCK_PROCESS_CPUTIME_ID, &tick);
  float t = float(tick.tv_nsec-start_tick.tv_nsec)/float(0x1FFFFFFF);
#endif

  static const glm::vec3 X(1.0f, 0.0f, 0.0f);
  static const glm::vec3 Y(0.0f, 1.0f, 0.0f);
  static const glm::vec3 Z(0.0f, 0.0f, 1.0f);

  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  // Activate shader program
  glUseProgram(cube_prog);
  glClear(GL_DEPTH_BUFFER_BIT);

  glm::mat4 tc_matrix =  glm::translate(glm::mat4(1.0f), glm::vec3(0.0f, 0.0f, -4.0f)) * 
   glm::scale(glm::mat4(1.0f),glm::vec3(1.5f)) * glm::rotate(glm::mat4(1.0f), 5.0f*t, Y) *
    glm::rotate(glm::mat4(1.0f), 5.0f*t, Z) * glm::rotate(glm::mat4(1.0f), 5.0f*t, X);
  glUniformMatrix4fv(mv_mat_loc, 1, GL_FALSE, glm::value_ptr(tc_matrix));
  tc_matrix = glm::perspective(30.0f, 1.0f/aspect, 0.1f, 10.0f);
  glUniformMatrix4fv(prj_mat_loc, 1, GL_FALSE, glm::value_ptr(tc_matrix));

  // Enable the vertex attributes for rendering
  glBindVertexArray(vao);
  glEnableVertexAttribArray(0);
  glEnableVertexAttribArray(1);
  glDrawArrays(GL_TRIANGLES, 0, 12*3);  

  glutSwapBuffers();
  glutPostRedisplay();
}

void Keyboard(unsigned char key, int x, int y)
{
  switch (key) {
  case 27:
    glUseProgram(0);
    glDeleteProgram(cube_prog);
    exit(0);
    break;
  default:
    break;
  }
}

void Reshape(int width, int height)
{
  glViewport(0, 0 , width, height);

  aspect = float(height) / float(width);
}

int main(int argc, char ** argv)                            
{                                                           
#ifdef _DEBUG
  glutInitContextFlags(GLUT_DEBUG);
#endif

  //! Connect OpenGL with a Window system
  glutInit(&argc, argv);
  glutInitContextProfile(GLUT_CORE_PROFILE);
  glutInitContextVersion(3, 1);
  glutInitWindowSize(1024, 768);
  glutInitWindowPosition (140, 140);
  glutInitDisplayMode(GLUT_RGBA | GLUT_DOUBLE | GLUT_DEPTH);
  glutCreateWindow("OpenGL Application");

  //! Specify callback functions
  glutDisplayFunc(Display);
  glutReshapeFunc(Reshape);
  glutKeyboardFunc(Keyboard);
  //glutIdleFunc(Display);
 
  //! Initialize glew
#ifdef WIN32
  glewInit();
#endif

#if defined _DEBUG && defined WIN32
  if (glDebugMessageCallbackARB != NULL)
    glDebugMessageCallbackARB(DebugOutputCallback, this);
#endif

  //! Initialize Opengl context
  Initialize(); 
                                
  //! Endless loop waiting for events
  glutMainLoop();                               
                                                            
  return 0;                                               
}                                                           
 
